/**
 * Converts a number to words (for invoice amounts)
 * Supports numbers up to 999,999,999.99
 */

const ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine'];
const teens = ['Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'];
const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];

function convertLessThanOneThousand(number) {
  if (number === 0) {
    return '';
  }

  let result = '';

  if (number >= 100) {
    result += ones[Math.floor(number / 100)] + ' Hundred ';
    number %= 100;
  }

  if (number >= 10 && number < 20) {
    result += teens[number - 10] + ' ';
  } else {
    if (number >= 20) {
      result += tens[Math.floor(number / 10)] + ' ';
      number %= 10;
    }
    if (number > 0) {
      result += ones[number] + ' ';
    }
  }

  return result.trim();
}

export function numberToWords(num) {
  if (num === 0) {
    return 'Zero';
  }

  // Handle negative numbers
  if (num < 0) {
    return 'Minus ' + numberToWords(Math.abs(num));
  }

  // Split into integer and decimal parts
  const parts = num.toFixed(2).split('.');
  const integerPart = parseInt(parts[0], 10);
  const decimalPart = parseInt(parts[1], 10);

  let result = '';

  // Handle millions
  if (integerPart >= 1000000) {
    result += convertLessThanOneThousand(Math.floor(integerPart / 1000000)) + ' Million ';
  }

  // Handle thousands
  const remainder = integerPart % 1000000;
  if (remainder >= 1000) {
    result += convertLessThanOneThousand(Math.floor(remainder / 1000)) + ' Thousand ';
  }

  // Handle hundreds, tens, and ones
  const lastPart = integerPart % 1000;
  if (lastPart > 0) {
    result += convertLessThanOneThousand(lastPart);
  }

  result = result.trim();

  // Add decimal part if exists
  if (decimalPart > 0) {
    result += ' and ' + decimalPart + '/100';
  }

  return result;
}

/**
 * Converts a number to words with currency
 * @param {number} amount - The amount to convert
 * @param {string} currency - Currency code (AED, USD, GBP, etc.)
 * @returns {string} - Amount in words with currency
 */
export function amountToWords(amount, currency = 'AED') {
  const currencyNames = {
    'USD': 'US Dollars',
    '$': 'US Dollars',
    'GBP': 'British Pounds',
    '£': 'British Pounds',
    'EUR': 'Euros',
    '€': 'Euros',
    'AED': 'Dirhams',
    'INR': 'Rupees',
    '₹': 'Rupees',
    'SAR': 'Saudi Riyals',
    'CAD': 'Canadian Dollars',
    'AUD': 'Australian Dollars'
  };

  const currencyName = currencyNames[currency] || currency;
  const words = numberToWords(amount);

  return `${words} ${currencyName} Only`;
}

export default { numberToWords, amountToWords };
