import express from 'express';
import moment from 'moment-timezone';
import Class from '../../models/Class.js';
import Student from '../../models/student/Student.js';
import { teacherAuth } from '../../middleware/auth.js';

const router = express.Router();

// Get teacher reports with date range
router.get('/', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { startDate, endDate } = req.query;

    // Default to current month if no dates provided
    const start = startDate
      ? moment(startDate).startOf('day').toDate()
      : moment().startOf('month').toDate();
    const end = endDate
      ? moment(endDate).endOf('day').toDate()
      : moment().endOf('month').toDate();

    // Get all classes for the teacher in the date range
    const classes = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: start,
        $lte: end
      }
    })
    .populate('studentId', 'studentName class')
    .sort({ scheduledDate: 1 });

    // Calculate summary statistics
    const totalClasses = classes.length;
    const completedClasses = classes.filter(c => c.status === 'completed').length;
    const cancelledClasses = classes.filter(c => c.status === 'cancelled').length;

    // Get unique students
    const uniqueStudents = [...new Set(classes.map(c => c.studentId?._id?.toString()).filter(Boolean))];
    const totalStudents = uniqueStudents.length;

    // Calculate attendance
    const completedClassesWithAttendance = classes.filter(c =>
      c.status === 'completed' && c.attendance?.markedAt
    );
    const attendedClasses = completedClassesWithAttendance.filter(c =>
      c.attendance?.present || c.attendance?.studentJoined
    ).length;
    const averageAttendance = completedClassesWithAttendance.length > 0
      ? (attendedClasses / completedClassesWithAttendance.length) * 100
      : 0;

    // Calculate homework statistics
    const homeworkAssigned = classes.filter(c => c.homework?.assigned).length;
    const homeworkCompleted = classes.filter(c => c.homework?.completed).length;

    // Student performance breakdown
    const studentPerformanceMap = {};

    classes.forEach(cls => {
      if (!cls.studentId) return;

      const studentId = cls.studentId._id.toString();

      if (!studentPerformanceMap[studentId]) {
        studentPerformanceMap[studentId] = {
          studentId: studentId,
          studentName: cls.studentId.studentName,
          class: cls.studentId.class,
          totalClasses: 0,
          attended: 0,
          homeworkAssigned: 0,
          homeworkCompleted: 0,
          grades: []
        };
      }

      const student = studentPerformanceMap[studentId];

      if (cls.status === 'completed') {
        student.totalClasses++;

        if (cls.attendance?.present || cls.attendance?.studentJoined) {
          student.attended++;
        }
      }

      if (cls.homework?.assigned) {
        student.homeworkAssigned++;
        if (cls.homework?.completed) {
          student.homeworkCompleted++;
        }
      }

      // Add grade if available (you can extend this based on your schema)
      if (cls.grade) {
        student.grades.push(cls.grade);
      }
    });

    // Format student performance
    const studentPerformance = Object.values(studentPerformanceMap).map(student => ({
      ...student,
      attendanceRate: student.totalClasses > 0
        ? (student.attended / student.totalClasses) * 100
        : 0,
      averageGrade: student.grades.length > 0
        ? student.grades.reduce((a, b) => a + b, 0) / student.grades.length
        : 0
    }));

    // Subject-wise breakdown
    const subjectMap = {};

    classes.forEach(cls => {
      if (!cls.subject) return;

      if (!subjectMap[cls.subject]) {
        subjectMap[cls.subject] = {
          subject: cls.subject,
          classes: 0,
          attended: 0,
          grades: []
        };
      }

      const subject = subjectMap[cls.subject];

      if (cls.status === 'completed') {
        subject.classes++;

        if (cls.attendance?.present || cls.attendance?.studentJoined) {
          subject.attended++;
        }
      }

      if (cls.grade) {
        subject.grades.push(cls.grade);
      }
    });

    const subjectBreakdown = Object.values(subjectMap).map(subject => ({
      subject: subject.subject,
      classes: subject.classes,
      attendance: subject.classes > 0
        ? Math.round((subject.attended / subject.classes) * 100)
        : 0,
      avgGrade: subject.grades.length > 0
        ? Math.round(subject.grades.reduce((a, b) => a + b, 0) / subject.grades.length)
        : 0
    }));

    // Monthly trends (last 3 months from end date)
    const monthlyTrends = [];
    for (let i = 2; i >= 0; i--) {
      const monthStart = moment(end).subtract(i, 'months').startOf('month').toDate();
      const monthEnd = moment(end).subtract(i, 'months').endOf('month').toDate();

      const monthClasses = await Class.find({
        teacherId: teacherId,
        scheduledDate: {
          $gte: monthStart,
          $lte: monthEnd
        },
        status: 'completed'
      });

      const monthAttended = monthClasses.filter(c =>
        c.attendance?.present || c.attendance?.studentJoined
      ).length;

      monthlyTrends.push({
        month: moment(monthStart).format('MMM YYYY'),
        classes: monthClasses.length,
        attendance: monthClasses.length > 0
          ? Math.round((monthAttended / monthClasses.length) * 100)
          : 0
      });
    }

    res.json({
      summary: {
        totalClasses,
        completedClasses,
        cancelledClasses,
        totalStudents,
        averageAttendance: Math.round(averageAttendance * 10) / 10,
        homeworkAssigned,
        homeworkCompleted
      },
      studentPerformance: studentPerformance.sort((a, b) =>
        b.attendanceRate - a.attendanceRate
      ),
      subjectBreakdown: subjectBreakdown.sort((a, b) =>
        b.classes - a.classes
      ),
      monthlyTrends
    });

  } catch (error) {
    console.error('Get teacher reports error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Download report as CSV
router.get('/download', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { startDate, endDate, format = 'csv' } = req.query;

    // Default to current month if no dates provided
    const start = startDate
      ? moment(startDate).startOf('day').toDate()
      : moment().startOf('month').toDate();
    const end = endDate
      ? moment(endDate).endOf('day').toDate()
      : moment().endOf('month').toDate();

    // Get all classes for the teacher in the date range
    const classes = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: start,
        $lte: end
      }
    })
    .populate('studentId', 'studentName class')
    .sort({ scheduledDate: 1 });

    if (format === 'csv') {
      // Generate CSV
      const csvRows = [];

      // Header
      csvRows.push('Teaching Report');
      csvRows.push(`Period: ${moment(start).format('YYYY-MM-DD')} to ${moment(end).format('YYYY-MM-DD')}`);
      csvRows.push('');

      // Summary Section
      csvRows.push('SUMMARY');
      csvRows.push('Metric,Value');

      const totalClasses = classes.length;
      const completedClasses = classes.filter(c => c.status === 'completed').length;
      const cancelledClasses = classes.filter(c => c.status === 'cancelled').length;
      const uniqueStudents = [...new Set(classes.map(c => c.studentId?._id?.toString()).filter(Boolean))];

      const completedClassesWithAttendance = classes.filter(c =>
        c.status === 'completed' && c.attendance?.markedAt
      );
      const attendedClasses = completedClassesWithAttendance.filter(c =>
        c.attendance?.present || c.attendance?.studentJoined
      ).length;
      const averageAttendance = completedClassesWithAttendance.length > 0
        ? (attendedClasses / completedClassesWithAttendance.length) * 100
        : 0;

      const homeworkAssigned = classes.filter(c => c.homework?.assigned).length;
      const homeworkCompleted = classes.filter(c => c.homework?.completed).length;

      csvRows.push(`Total Classes,${totalClasses}`);
      csvRows.push(`Completed Classes,${completedClasses}`);
      csvRows.push(`Cancelled Classes,${cancelledClasses}`);
      csvRows.push(`Total Students,${uniqueStudents.length}`);
      csvRows.push(`Average Attendance,${averageAttendance.toFixed(1)}%`);
      csvRows.push(`Homework Assigned,${homeworkAssigned}`);
      csvRows.push(`Homework Completed,${homeworkCompleted}`);
      csvRows.push('');

      // Detailed Classes
      csvRows.push('DETAILED CLASS LIST');
      csvRows.push([
        'Date',
        'Time',
        'Student',
        'Class',
        'Subject',
        'Duration (min)',
        'Status',
        'Attendance',
        'Homework'
      ].join(','));

      classes.forEach(cls => {
        const date = moment(cls.scheduledDate).format('YYYY-MM-DD');
        const time = cls.startTime || moment(cls.scheduledDate).format('HH:mm');
        const student = cls.studentId?.studentName || 'Unknown';
        const studentClass = cls.studentId?.class || 'N/A';
        const subject = cls.subject || 'N/A';
        const duration = cls.duration || 60;
        const status = cls.status || 'scheduled';
        const attendance = cls.attendance?.present ? 'Present' :
                          cls.attendance?.late ? 'Late' :
                          cls.attendance?.absent ? 'Absent' : 'Not Marked';
        const homework = cls.homework?.assigned ?
                        (cls.homework?.completed ? 'Completed' : 'Assigned') : 'None';

        csvRows.push([
          date,
          time,
          `"${student}"`,
          `"${studentClass}"`,
          `"${subject}"`,
          duration,
          status,
          attendance,
          homework
        ].join(','));
      });

      csvRows.push('');

      // Student Performance Summary
      const studentPerformanceMap = {};
      classes.forEach(cls => {
        if (!cls.studentId) return;
        const studentId = cls.studentId._id.toString();

        if (!studentPerformanceMap[studentId]) {
          studentPerformanceMap[studentId] = {
            studentName: cls.studentId.studentName,
            class: cls.studentId.class,
            totalClasses: 0,
            attended: 0,
            homeworkAssigned: 0,
            homeworkCompleted: 0
          };
        }

        const student = studentPerformanceMap[studentId];
        if (cls.status === 'completed') {
          student.totalClasses++;
          if (cls.attendance?.present || cls.attendance?.studentJoined) {
            student.attended++;
          }
        }
        if (cls.homework?.assigned) {
          student.homeworkAssigned++;
          if (cls.homework?.completed) {
            student.homeworkCompleted++;
          }
        }
      });

      csvRows.push('STUDENT PERFORMANCE SUMMARY');
      csvRows.push('Student,Class,Total Classes,Attended,Attendance Rate,Homework Assigned,Homework Completed');

      Object.values(studentPerformanceMap).forEach(student => {
        const attendanceRate = student.totalClasses > 0
          ? ((student.attended / student.totalClasses) * 100).toFixed(1)
          : '0.0';

        csvRows.push([
          `"${student.studentName}"`,
          `"${student.class}"`,
          student.totalClasses,
          student.attended,
          `${attendanceRate}%`,
          student.homeworkAssigned,
          student.homeworkCompleted
        ].join(','));
      });

      const csv = csvRows.join('\n');

      // Generate filename
      const teacherName = req.user.profile?.name?.replace(/\s+/g, '_') || 'Teacher';
      const dateStr = `${moment(start).format('YYYY-MM-DD')}_to_${moment(end).format('YYYY-MM-DD')}`;
      const filename = `Teaching_Report_${teacherName}_${dateStr}.csv`;

      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send(csv);
    } else {
      res.status(400).json({ message: 'Unsupported format. Use format=csv' });
    }
  } catch (error) {
    console.error('Download report error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

export default router;