import express from 'express';
import User from '../../models/admin/User.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import TeacherClass from '../../models/Class.js';
import Invoice from '../../models/admin/Invoice.js';
import { adminAuth } from '../../middleware/auth.js';
import PDFDocument from 'pdfkit';

const router = express.Router();

// Get students by country statistics
router.get('/students-by-country', adminAuth, async (req, res) => {
  try {
    const studentsByCountry = await Student.aggregate([
      {
        $match: {
          isActive: true,
          isDeleted: false
        }
      },
      {
        $group: {
          _id: '$country',
          count: { $sum: 1 }
        }
      },
      {
        $sort: { count: -1 }
      }
    ]);

    // Map country names to emoji flags and colors
    const countryConfig = {
      'UAE': { flag: '🇦🇪', color: '#1976D2' },
      'Saudi Arabia': { flag: '🇸🇦', color: '#009739' },
      'Qatar': { flag: '🇶🇦', color: '#8E0051' },
      'Kuwait': { flag: '🇰🇼', color: '#007A33' },
      'Bahrain': { flag: '🇧🇭', color: '#DC143C' },
      'Oman': { flag: '🇴🇲', color: '#DC143C' },
      'India': { flag: '🇮🇳', color: '#FF6600' },
      'UK': { flag: '🇬🇧', color: '#1976D2' },
      'Australia': { flag: '🇦🇺', color: '#00008B' },
      'New Zealand': { flag: '🇳🇿', color: '#1976D2' }
    };

    const formattedData = studentsByCountry.map(item => ({
      country: item._id,
      count: item.count,
      flag: countryConfig[item._id]?.flag || '🌍',
      color: countryConfig[item._id]?.color || '#1976D2'
    }));

    res.json(formattedData);
  } catch (error) {
    console.error('Error fetching students by country:', error);
    res.status(500).json({ error: 'Failed to fetch students by country' });
  }
});

// Get dashboard statistics
router.get('/stats', adminAuth, async (req, res) => {
  try {
    console.log('📊 Stats endpoint called by user:', req.user?.username || req.user?._id);

    // Currency conversion rates to USD
    const currencyToUSD = {
      'USD': 1,
      '$': 1,
      'GBP': 1.27,
      '£': 1.27,
      'EUR': 1.09,
      '€': 1.09,
      'AED': 0.27,
      'INR': 0.012,
      '₹': 0.012,
      'SAR': 0.27,
      'CAD': 0.74,
      'AUD': 0.65
    };

    const [totalStudents, totalTeachers, totalClasses, invoices] = await Promise.all([
      Student.countDocuments({
        $or: [
          { isDeleted: false },
          { isDeleted: { $exists: false } }
        ]
      }),
      Teacher.countDocuments({
        $or: [
          { isDeleted: false },
          { isDeleted: { $exists: false } }
        ]
      }),
      TeacherClass.countDocuments({ isDeleted: { $ne: true } }),
      Invoice.find({ isDeleted: false })
    ]);

    const totalRevenue = invoices.reduce((sum, invoice) => {
      if (invoice.status === 'paid') {
        const currency = invoice.currency || 'USD';
        const conversionRate = currencyToUSD[currency] || 1;
        return sum + (invoice.totalAmount * conversionRate);
      }
      return sum;
    }, 0);

    console.log('📊 Stats calculated:', {
      totalStudents,
      totalTeachers,
      totalClasses,
      totalRevenue
    });

    res.json({
      totalStudents,
      totalTeachers,
      totalClasses,
      totalRevenue
    });
  } catch (error) {
    console.error('❌ Error fetching stats:', error);
    console.error('Error stack:', error.stack);
    res.status(500).json({
      error: 'Failed to fetch statistics',
      details: process.env.NODE_ENV !== 'production' ? error.message : undefined
    });
  }
});

// Get students report
router.get('/students', adminAuth, async (req, res) => {
  try {
    const { startDate, endDate, page = 1, limit = 10 } = req.query;
    const query = {
      $or: [
        { isDeleted: false },
        { isDeleted: { $exists: false } }
      ]
    };

    if (startDate && endDate) {
      query.createdAt = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);
    const total = await Student.countDocuments(query);

    const students = await Student.find(query)
      .populate('userId', 'isActive')
      .skip(skip)
      .limit(parseInt(limit))
      .lean();

    console.log(`📚 Fetched ${students.length} students (page ${page}, total: ${total})`);
    console.log(`📚 Query used:`, JSON.stringify(query));

    // Get class counts and attendance for each student
    const studentsWithClasses = await Promise.all(
      students.map(async (student) => {
        const classes = await TeacherClass.find({
          studentId: student._id,
          isDeleted: { $ne: true }
        }).select('attendanceStatus.studentAttended status');

        const totalClasses = classes.length;
        const attendedClasses = classes.filter(cls =>
          cls.attendanceStatus?.studentAttended === true
        ).length;

        const attendancePercentage = totalClasses > 0
          ? Math.round((attendedClasses / totalClasses) * 100)
          : 0;

        return {
          id: student._id,
          name: student.studentName,
          email: student.email,
          classes: totalClasses,
          attendance: `${attendancePercentage}%`,
          status: student.isActive ? 'Active' : 'Inactive'
        };
      })
    );

    res.json({
      data: studentsWithClasses,
      totalPages: Math.ceil(total / parseInt(limit)),
      currentPage: parseInt(page),
      total
    });
  } catch (error) {
    console.error('Error fetching students report:', error);
    res.status(500).json({ error: 'Failed to fetch students report' });
  }
});

// Get teachers report
router.get('/teachers', adminAuth, async (req, res) => {
  try {
    const { startDate, endDate, page = 1, limit = 10 } = req.query;
    const query = {
      $or: [
        { isDeleted: false },
        { isDeleted: { $exists: false } }
      ]
    };

    if (startDate && endDate) {
      query.createdAt = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);
    const total = await Teacher.countDocuments(query);

    const teachers = await Teacher.find(query)
      .populate('userId', 'isActive')
      .skip(skip)
      .limit(parseInt(limit))
      .lean();

    console.log(`👨‍🏫 Fetched ${teachers.length} teachers (page ${page}, total: ${total})`);
    console.log(`👨‍🏫 Query used:`, JSON.stringify(query));

    // Get class counts and student counts for each teacher
    const teachersWithStats = await Promise.all(
      teachers.map(async (teacher) => {
        const classes = await TeacherClass.countDocuments({
          teacherId: teacher._id,
          isDeleted: { $ne: true }
        });

        // Get unique students for this teacher
        const uniqueStudents = await TeacherClass.distinct('studentId', {
          teacherId: teacher._id,
          isDeleted: { $ne: true }
        });

        // Calculate average rating based on completed classes
        const completedClasses = await TeacherClass.countDocuments({
          teacherId: teacher._id,
          status: 'completed',
          isDeleted: { $ne: true }
        });

        // Simple rating calculation: more completed classes = higher rating (4.0-5.0)
        const rating = completedClasses > 0
          ? Math.min(4.0 + (completedClasses / 100), 5.0).toFixed(1)
          : '4.0';

        return {
          id: teacher._id,
          name: teacher.name,
          subject: teacher.subjects?.join(', ') || 'General',
          students: uniqueStudents.length,
          classes: classes,
          rating: parseFloat(rating)
        };
      })
    );

    res.json({
      data: teachersWithStats,
      totalPages: Math.ceil(total / parseInt(limit)),
      currentPage: parseInt(page),
      total
    });
  } catch (error) {
    console.error('Error fetching teachers report:', error);
    res.status(500).json({ error: 'Failed to fetch teachers report' });
  }
});

// Get classes report
router.get('/classes', adminAuth, async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const query = { isDeleted: { $ne: true } };

    if (startDate && endDate) {
      query.scheduledDate = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const classes = await TeacherClass.find(query)
      .populate('teacherId', 'name')
      .populate('studentId', 'studentName')
      .select('subject teacherId studentId duration status scheduledDate')
      .sort({ scheduledDate: -1 })
      .lean();
    const classesReport = classes.map(cls => {
      const status = cls.status || 'scheduled';
      const capitalizedStatus = status.charAt(0).toUpperCase() + status.slice(1);

      return {
        id: cls._id,
        subject: cls.subject || 'General',
        teacher: cls.teacherId?.name || 'Unknown',
        students: cls.studentId ? 1 : 0,
        date: cls.scheduledDate ? new Date(cls.scheduledDate).toLocaleDateString() : 'Not scheduled',
        duration: cls.duration ? `${cls.duration} min` : '60 min',
        status: capitalizedStatus
      };
    });

    res.json(classesReport);
  } catch (error) {
    console.error('Error fetching classes report:', error);
    res.status(500).json({ error: 'Failed to fetch classes report' });
  }
});

// Get finance report
router.get('/finance', adminAuth, async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const query = {};

    if (startDate && endDate) {
      query.createdAt = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const invoices = await Invoice.find(query)
      .populate('studentId', 'studentName')
      .select('studentId totalAmount dueDate parentName email status currency')
      .sort({ createdAt: -1 })
      .lean();
        
      
      

    const financeReport = invoices.map(invoice => {
      const status = invoice.status || 'draft';
      const capitalizedStatus = status.charAt(0).toUpperCase() + status.slice(1);

      return {
        id: invoice._id,
        student: invoice.studentId?.studentName || 'Unknown',
        amount: `${invoice.currency || 'USD'} ${invoice.totalAmount || 0}`,
        dueDate: invoice.dueDate ? new Date(invoice.dueDate).toLocaleDateString() : 'No due date',
        status: capitalizedStatus,
        method: 'Bank Transfer'
      };
    });

    res.json(financeReport);
  } catch (error) {
    console.error('Error fetching finance report:', error);
    res.status(500).json({ error: 'Failed to fetch finance report' });
  }
});

// Export report data
router.get('/export/:reportType', adminAuth, async (req, res) => {
  try {
    const { reportType } = req.params;
    const { startDate, endDate, format } = req.query;

    let reportData;
    let fileName;

    // Fetch data based on report type
    switch (reportType) {
      case 'students':
        reportData = await getStudentsReportData(startDate, endDate);
        fileName = 'students_report';
        break;
      case 'teachers':
        reportData = await getTeachersReportData(startDate, endDate);
        fileName = 'teachers_report';
        break;
      case 'classes':
        reportData = await getClassesReportData(startDate, endDate);
        fileName = 'classes_report';
        break;
      case 'finance':
        reportData = await getFinanceReportData(startDate, endDate);
        fileName = 'finance_report';
        break;
      case 'attendance':
        reportData = await getAttendanceReportData(startDate, endDate);
        fileName = 'attendance_report';
        break;
      default:
        return res.status(400).json({ error: 'Invalid report type' });
    }

    if (format === 'pdf') {
      // Generate PDF
      const pdfBuffer = await generatePDF(reportData, reportType, fileName, startDate, endDate);
      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="${fileName}.pdf"`);
      res.send(pdfBuffer);
    } else if (format === 'csv' || format === 'excel') {
      // Convert to CSV format
      const csv = convertToCSV(reportData);
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${fileName}.csv"`);
      res.send(csv);
    } else if (format === 'json') {
      res.setHeader('Content-Type', 'application/json');
      res.setHeader('Content-Disposition', `attachment; filename="${fileName}.json"`);
      res.json(reportData);
    } else {
      res.status(400).json({ error: 'Invalid export format' });
    }
  } catch (error) {
    console.error('Error exporting report:', error);
    res.status(500).json({ error: 'Failed to export report' });
  }
});

// Helper function to convert data to CSV
function convertToCSV(data) {
  if (!data || data.length === 0) {
    return '';
  }

  const headers = Object.keys(data[0]).filter(key => key !== 'id');
  const csvHeaders = headers.join(',');

  const csvRows = data.map(row => {
    return headers.map(header => {
      const value = row[header];
      // Escape commas and quotes
      if (typeof value === 'string' && (value.includes(',') || value.includes('"'))) {
        return `"${value.replace(/"/g, '""')}"`;
      }
      return value;
    }).join(',');
  });

  return [csvHeaders, ...csvRows].join('\n');
}

// Helper function to generate PDF
async function generatePDF(data, reportType, fileName, startDate, endDate) {
  return new Promise((resolve, reject) => {
    try {
      const doc = new PDFDocument({ margin: 50 });
      const chunks = [];

      doc.on('data', chunk => chunks.push(chunk));
      doc.on('end', () => resolve(Buffer.concat(chunks)));
      doc.on('error', reject);

      // Header
      doc.fontSize(20).text('EduMetrix Report', { align: 'center' });
      doc.moveDown(0.5);

      // Report Title
      const reportTitles = {
        'students': 'Student Report',
        'teachers': 'Teacher Report',
        'classes': 'Class Report',
        'finance': 'Financial Report',
        'attendance': 'Attendance Report'
      };

      doc.fontSize(16).text(reportTitles[reportType] || 'Report', { align: 'center' });
      doc.moveDown(0.5);

      // Date Range
      if (startDate && endDate) {
        doc.fontSize(10).text(
          `Period: ${new Date(startDate).toLocaleDateString()} - ${new Date(endDate).toLocaleDateString()}`,
          { align: 'center' }
        );
      }

      doc.fontSize(10).text(`Generated on: ${new Date().toLocaleString()}`, { align: 'center' });
      doc.moveDown(1);

      // Draw line
      doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
      doc.moveDown(1);

      if (!data || data.length === 0) {
        doc.fontSize(12).text('No data available for the selected criteria', { align: 'center' });
        doc.end();
        return;
      }

      // Get headers (excluding id)
      const headers = Object.keys(data[0]).filter(key => key !== 'id');

      // Calculate column widths
      const pageWidth = 500;
      const columnWidth = pageWidth / headers.length;

      // Table Header
      doc.fontSize(10).font('Helvetica-Bold');
      let xPos = 50;
      headers.forEach(header => {
        doc.text(header.charAt(0).toUpperCase() + header.slice(1), xPos, doc.y, {
          width: columnWidth,
          align: 'left'
        });
        xPos += columnWidth;
      });

      doc.moveDown(0.5);
      doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
      doc.moveDown(0.5);

      // Table Rows
      doc.font('Helvetica').fontSize(9);

      data.forEach((row, index) => {
        // Check if we need a new page
        if (doc.y > 700) {
          doc.addPage();
          doc.fontSize(10).font('Helvetica-Bold');
          xPos = 50;
          headers.forEach(header => {
            doc.text(header.charAt(0).toUpperCase() + header.slice(1), xPos, doc.y, {
              width: columnWidth,
              align: 'left'
            });
            xPos += columnWidth;
          });
          doc.moveDown(0.5);
          doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
          doc.moveDown(0.5);
          doc.font('Helvetica').fontSize(9);
        }

        const startY = doc.y;
        xPos = 50;

        headers.forEach(header => {
          const value = row[header];
          const text = value !== null && value !== undefined ? String(value) : '';
          doc.text(text, xPos, startY, {
            width: columnWidth - 5,
            align: 'left',
            ellipsis: true
          });
          xPos += columnWidth;
        });

        doc.moveDown(0.8);

        // Draw separator line every 5 rows
        if ((index + 1) % 5 === 0) {
          doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke('#CCCCCC');
          doc.moveDown(0.3);
        }
      });

      // Footer
      const pageCount = doc.bufferedPageRange().count;
      for (let i = 0; i < pageCount; i++) {
        doc.switchToPage(i);
        doc.fontSize(8).text(
          `Page ${i + 1} of ${pageCount}`,
          50,
          doc.page.height - 50,
          { align: 'center' }
        );
      }

      doc.end();
    } catch (error) {
      reject(error);
    }
  });
}

// Helper functions to get report data
async function getStudentsReportData(startDate, endDate) {
  const query = { isDeleted: false };
  if (startDate && endDate) {
    query.createdAt = {
      $gte: new Date(startDate),
      $lte: new Date(endDate)
    };
  }

  const students = await Student.find(query)
    .populate('userId', 'isActive')
    .lean();

  return await Promise.all(
    students.map(async (student) => {
      const classes = await TeacherClass.find({
        studentId: student._id,
        isDeleted: { $ne: true }
      }).select('attendanceStatus.studentAttended status');

      const totalClasses = classes.length;
      const attendedClasses = classes.filter(cls =>
        cls.attendanceStatus?.studentAttended === true
      ).length;

      const attendancePercentage = totalClasses > 0
        ? Math.round((attendedClasses / totalClasses) * 100)
        : 0;

      return {
        id: student._id,
        name: student.studentName,
        email: student.email,
        classes: totalClasses,
        attendance: `${attendancePercentage}%`,
        status: student.userId?.isActive ? 'Active' : 'Inactive'
      };
    })
  );
}

async function getTeachersReportData(startDate, endDate) {
  const query = { isDeleted: false };
  if (startDate && endDate) {
    query.createdAt = {
      $gte: new Date(startDate),
      $lte: new Date(endDate)
    };
  }

  const teachers = await Teacher.find(query)
    .populate('userId', 'isActive')
    .lean();

  return await Promise.all(
    teachers.map(async (teacher) => {
      const classes = await TeacherClass.countDocuments({
        teacherId: teacher._id,
        isDeleted: { $ne: true }
      });

      const uniqueStudents = await TeacherClass.distinct('studentId', {
        teacherId: teacher._id,
        isDeleted: { $ne: true }
      });

      const completedClasses = await TeacherClass.countDocuments({
        teacherId: teacher._id,
        status: 'completed',
        isDeleted: { $ne: true }
      });

      const rating = completedClasses > 0
        ? Math.min(4.0 + (completedClasses / 100), 5.0).toFixed(1)
        : '4.0';

      return {
        id: teacher._id,
        name: teacher.name,
        subject: teacher.subjects?.join(', ') || 'General',
        students: uniqueStudents.length,
        classes: classes,
        rating: parseFloat(rating)
      };
    })
  );
}

async function getClassesReportData(startDate, endDate) {
  const query = { isDeleted: { $ne: true } };
  if (startDate && endDate) {
    query.scheduledDate = {
      $gte: new Date(startDate),
      $lte: new Date(endDate)
    };
  }

  const classes = await TeacherClass.find(query)
    .populate('teacherId', 'name')
    .populate('studentId', 'studentName')
    .select('subject teacherId studentId duration status scheduledDate')
    .sort({ scheduledDate: -1 })
    .lean();

  return classes.map(cls => {
    const status = cls.status || 'scheduled';
    const capitalizedStatus = status.charAt(0).toUpperCase() + status.slice(1);

    return {
      id: cls._id,
      subject: cls.subject || 'General',
      teacher: cls.teacherId?.name || 'Unknown',
      students: cls.studentId ? 1 : 0,
      date: cls.scheduledDate ? new Date(cls.scheduledDate).toLocaleDateString() : 'Not scheduled',
      duration: cls.duration ? `${cls.duration} min` : '60 min',
      status: capitalizedStatus
    };
  });
}

async function getFinanceReportData(startDate, endDate) {
  const query = {};
  if (startDate && endDate) {
    query.createdAt = {
      $gte: new Date(startDate),
      $lte: new Date(endDate)
    };
  }

  const invoices = await Invoice.find(query)
    .populate('studentId', 'studentName')
    .select('studentId totalAmount dueDate status currency')
    .sort({ createdAt: -1 })
    .lean();

  return invoices.map(invoice => {
    const status = invoice.status || 'draft';
    const capitalizedStatus = status.charAt(0).toUpperCase() + status.slice(1);

    return {
      id: invoice._id,
      student: invoice.studentId?.studentName || 'Unknown',
      amount: `${invoice.currency || 'USD'} ${invoice.totalAmount || 0}`,
      dueDate: invoice.dueDate ? new Date(invoice.dueDate).toLocaleDateString() : 'No due date',
      status: capitalizedStatus,
      method: 'Bank Transfer'
    };
  });
}

async function getAttendanceReportData(startDate, endDate) {
  const students = await Student.find({
    isDeleted: false,
    isActive: true
  }).select('studentName').lean();

  return await Promise.all(
    students.map(async (student) => {
      const query = {
        studentId: student._id,
        isDeleted: { $ne: true }
      };

      if (startDate && endDate) {
        query.scheduledDate = {
          $gte: new Date(startDate),
          $lte: new Date(endDate)
        };
      }

      const classes = await TeacherClass.find(query)
        .select('attendanceStatus.studentAttended status')
        .lean();

      const totalClasses = classes.length;
      const classesAttended = classes.filter(cls =>
        cls.attendanceStatus?.studentAttended === true
      ).length;

      const percentage = totalClasses > 0
        ? Math.round((classesAttended / totalClasses) * 100)
        : 0;

      let trend = 'Stable';
      if (percentage >= 90) trend = 'Up';
      else if (percentage < 75) trend = 'Down';

      return {
        id: student._id,
        student: student.studentName,
        classesAttended,
        totalClasses,
        percentage,
        trend
      };
    })
  );
}

// Get attendance report
router.get('/attendance', adminAuth, async (req, res) => {
  try {
    const { startDate, endDate } = req.query;

    // Get all active students
    const students = await Student.find({
      isDeleted: false,
      isActive: true
    }).select('studentName').lean();

    // For each student, calculate attendance
    const attendanceReport = await Promise.all(
      students.map(async (student) => {
        const query = {
          studentId: student._id,
          isDeleted: { $ne: true }
        };

        // Add date filter if provided
        if (startDate && endDate) {
          query.scheduledDate = {
            $gte: new Date(startDate),
            $lte: new Date(endDate)
          };
        }

        const classes = await TeacherClass.find(query)
          .select('attendanceStatus.studentAttended status')
          .lean();
          console.log('classes student>>',classes)

        const totalClasses = classes.length;
        const classesAttended = classes.filter(cls =>
          cls.attendanceStatus?.studentAttended === true
        ).length;

        const percentage = totalClasses > 0
          ? Math.round((classesAttended / totalClasses) * 100)
          : 0;

        // Determine trend based on percentage
        let trend = 'Stable';
        if (percentage >= 90) trend = 'Up';
        else if (percentage < 75) trend = 'Down';

        return {
          id: student._id,
          student: student.studentName,
          classesAttended,
          totalClasses,
          percentage,
          trend
        };
      })
    );

    res.json(attendanceReport);
  } catch (error) {
    console.error('Error fetching attendance report:', error);
    res.status(500).json({ error: 'Failed to fetch attendance report' });
  }
});

export default router;