import express from 'express';
import moment from 'moment';
import Class from '../../models/Class.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import Invoice from '../../models/admin/Invoice.js';
import TrialRequest from '../../models/admin/TrialRequest.js';
import RescheduleRequest from '../../models/shared/RescheduleRequest.js';
import { adminAuth } from '../../middleware/auth.js';

const router = express.Router();

router.get('/student-monthly-report/:studentId', adminAuth, async (req, res) => {
  try {
    const { month, year } = req.query;
    const studentId = req.params.studentId;
    
    if (!month || !year) {
      return res.status(400).json({ message: 'Month and year are required' });
    }

    const student = await Student.findById(studentId);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    const startDate = moment(`${year}-${month}-01`).startOf('month').toDate();
    const endDate = moment(`${year}-${month}-01`).endOf('month').toDate();

    const classes = await Class.find({
      studentId,
      scheduledDate: { $gte: startDate, $lte: endDate },
      // status: 'completed',
      // 'attendanceStatus.studentAttended': true
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const totalClasses = classes.length;
    const totalHours = classes.reduce((sum, cls) => sum + (cls.duration / 60), 0);
    const totalAmount = totalHours * student.hourlyRate;

    const classDetails = classes.map(cls => ({
      date: moment(cls.scheduledDate).format('YYYY-MM-DD'),
      subject: cls.subject,
      teacher: cls.teacherId.name,
      duration: cls.duration,
      rate: student.hourlyRate,
      amount: (cls.duration / 60) * student.hourlyRate
    }));

    res.json({
      student: {
        name: student.studentName,
        hourlyRate: student.hourlyRate,
        currency: student.currency
      },
      month: moment(`${year}-${month}-01`).format('MMMM YYYY'),
      summary: {
        totalClasses,
        totalHours: totalHours.toFixed(2),
        totalAmount: totalAmount.toFixed(2)
      },
      classes: classDetails
    });
  } catch (error) {
    console.error('Get student monthly report error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teacher-monthly-report/:teacherId', adminAuth, async (req, res) => {
  try {
    const { month, year } = req.query;
    const teacherId = req.params.teacherId;
    
    if (!month || !year) {
      return res.status(400).json({ message: 'Month and year are required' });
    }

    const teacher = await Teacher.findById(teacherId);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    const startDate = moment(`${year}-${month}-01`).startOf('month').toDate();
    const endDate = moment(`${year}-${month}-01`).endOf('month').toDate();

    const classes = await Class.find({
      teacherId,
      scheduledDate: { $gte: startDate, $lte: endDate },
      status: 'completed',
      'attendanceStatus.teacherAttended': true
    })
    .populate('studentId', 'studentName hourlyRate currency')
    .sort({ scheduledDate: 1 });

    const totalClasses = classes.length;
    const totalHours = classes.reduce((sum, cls) => sum + (cls.duration / 60), 0);

    let potentialIncome = 0;
    let salary = 0;

    if (teacher.salaryType === 'hourly') {
      potentialIncome = totalHours * teacher.salary.amount;
      salary = potentialIncome;
    } else {
      salary = teacher.salary.amount;
      potentialIncome = salary;
    }

    const classDetails = classes.map(cls => ({
      date: moment(cls.scheduledDate).format('YYYY-MM-DD'),
      student: cls.studentId.studentName,
      subject: cls.subject,
      duration: cls.duration,
      studentRate: cls.studentId.hourlyRate,
      studentCurrency: cls.studentId.currency
    }));

    const classesPerStudent = classes.reduce((acc, cls) => {
      const studentName = cls.studentId.studentName;
      if (!acc[studentName]) {
        acc[studentName] = 0;
      }
      acc[studentName]++;
      return acc;
    }, {});

    res.json({
      teacher: {
        name: teacher.name,
        salaryType: teacher.salaryType,
        salaryAmount: teacher.salary.amount,
        currency: teacher.salary.currency
      },
      month: moment(`${year}-${month}-01`).format('MMMM YYYY'),
      summary: {
        totalClasses,
        totalHours: totalHours.toFixed(2),
        salary: salary.toFixed(2),
        potentialIncome: potentialIncome.toFixed(2),
        classesPerStudent
      },
      classes: classDetails
    });
  } catch (error) {
    console.error('Get teacher monthly report error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/generate-invoice/:studentId', adminAuth, async (req, res) => {
  try {
    let { month, year, dueDate, discount, discountType, subtotal, paymentLink, paymentTerms, bankAccountDetails, companyDetails } = req.body;
    const { studentId } = req.params; // ✅ declare early

    // Support combined YYYY-MM (e.g. "2025-09")
    if (month && month.includes('-')) {
      [year, month] = month.split('-');
    }

    if (!month || !year || !dueDate) {
      return res.status(400).json({ message: 'Month, year, and due date are required' });
    }

    const student = await Student.findById(studentId);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    const existingInvoice = await Invoice.findOne({
      studentId,
      invoiceMonth: month,
      invoiceYear: parseInt(year, 10)
    });

    if (existingInvoice) {
      return res.status(400).json({ message: 'Invoice already exists for this month' });
    }

    const startDate = moment(`${year}-${month}-01`).startOf('month').toDate();
    const endDate = moment(`${year}-${month}-01`).endOf('month').toDate();

    const classes = await Class.find({
      studentId,
      scheduledDate: { $gte: startDate, $lte: endDate },
      // status: 'completed',
      // 'attendanceStatus.studentAttended': true
    })
      .populate('teacherId', 'name')
      .sort({ scheduledDate: 1 });

    if (classes.length === 0) {
      return res.status(400).json({ message: 'No completed classes found for this month' });
    }

    // Generate sequential invoice number: INV-2025-001, INV-2025-002, etc.
    const lastInvoice = await Invoice.findOne({ invoiceYear: parseInt(year, 10) })
      .sort({ createdAt: -1 })
      .select('invoiceNumber');

    let sequenceNumber = 1;
    if (lastInvoice && lastInvoice.invoiceNumber) {
      const match = lastInvoice.invoiceNumber.match(/INV-\d{4}-(\d+)/);
      if (match) {
        sequenceNumber = parseInt(match[1], 10) + 1;
      }
    }

    const invoiceNumber = `INV-${year}-${sequenceNumber.toString().padStart(3, '0')}`;

    const invoiceClasses = classes.map(cls => ({
      classId: cls._id,
      date: cls.scheduledDate,
      subject: cls.subject,
      teacher: cls.teacherId.name,
      duration: cls.duration,
      rate: student.hourlyRate,
      amount: (cls.duration / 60) * student.hourlyRate
    }));

    // Use subtotal from frontend if provided, otherwise calculate
    const calculatedSubtotal = subtotal || invoiceClasses.reduce((sum, cls) => sum + cls.amount, 0);

    // Use discount from frontend if provided
    const discountAmount = discount || 0;

    // No tax calculation
    const tax = 0;
    const totalAmount = calculatedSubtotal - discountAmount + tax;

    // Create items array for the invoice
    const items = invoiceClasses.map(cls => ({
      description: `${cls.subject} - ${cls.teacher} (${cls.duration} mins)`,
      quantity: 1,
      rate: cls.amount,
      amount: cls.amount
    }));

    const invoice = new Invoice({
      studentId,
      invoiceNumber,
      invoiceMonth: month,
      invoiceYear: parseInt(year, 10),
      date: new Date(),
      classes: invoiceClasses,
      items,
      totalClasses: classes.length,
      subtotal: parseFloat(calculatedSubtotal.toFixed(2)),
      discount: parseFloat(discountAmount.toFixed(2)),
      discountType: discountType || 'percentage',
      tax: parseFloat(tax.toFixed(2)),
      totalAmount: parseFloat(totalAmount.toFixed(2)),
      currency: student.currency,
      dueDate: moment(dueDate).toDate(),
      status: 'draft',
      enabled: true,
      paymentLink: paymentLink || '',
      paymentTerms: paymentTerms || '',
      bankAccountDetails: bankAccountDetails || {},
      companyDetails: companyDetails || {
        companyName: 'EduMetrix UK',
        email: 'contact@edumetrix.uk'
      }
    });

    await invoice.save();

    // Create notification for student
    try {
      const io = req.app.get('io');
      const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');

      if (io) {
        // Get currency symbol
        const getCurrencySymbol = (currency) => {
          const symbols = {
            'USD': '$', 'GBP': '£', 'EUR': '€', 'AED': 'AED',
            'INR': '₹', 'SAR': 'SAR', 'CAD': 'CA$', 'AUD': 'A$'
          };
          return symbols[currency] || currency || 'AED';
        };

        const currencySymbol = getCurrencySymbol(student.currency);
        const formattedAmount = `${currencySymbol} ${totalAmount.toFixed(2)}`;

        const notificationData = NotificationTemplates.invoiceGenerated(
          formattedAmount,
          moment(dueDate).format('MMM DD, YYYY')
        );

        await createNotification({
          userId: studentId, // Use the student's MongoDB _id
          userType: 'student',
          type: notificationData.type,
          title: notificationData.title,
          message: notificationData.message,
          relatedId: invoice._id,
          relatedModel: 'Invoice',
          priority: notificationData.priority,
          actionUrl: '/student/invoices',
          io
        });

        console.log(`📢 Sent invoice generated notification to student ${studentId}`);
      } else {
        console.warn('⚠️ Socket.IO instance not available for invoice notification');
      }
    } catch (notificationError) {
      console.error('Failed to send invoice notification:', notificationError);
      // Don't fail the invoice generation if notification fails
    }

    res.status(201).json({
      message: 'Invoice generated successfully',
      invoice: await Invoice.findById(invoice._id).populate('studentId', 'studentName parentName email')
    });
  } catch (error) {
    console.error('Generate invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});


router.get('/invoices', adminAuth, async (req, res) => {
  try {
    const { page = 1, limit = 10, status = 'all', month, year, studentId } = req.query;

    let query = {};
    if (status !== 'all') {
      query.status = status;
    }
    if (month && year) {
      query.invoiceMonth = month;
      query.invoiceYear = parseInt(year);
    }
    if (studentId) {
      query.studentId = studentId;
    }

    const invoices = await Invoice.find(query)
      .populate('studentId', 'studentName parentName email currency')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await Invoice.countDocuments(query);

    res.json({
      invoices,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get invoices error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/invoices/:id', adminAuth, async (req, res) => {
  try {
    const invoice = await Invoice.findById(req.params.id)
      .populate('studentId', 'studentName parentName email phoneNumber currency country');

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    res.json({ invoice });
  } catch (error) {
    console.error('Get invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/invoices/:id', adminAuth, async (req, res) => {
  try {
    const { items, dueDate, notes, subtotal, discount, discountType, tax, totalAmount } = req.body;

    const updateData = {};
    if (items) updateData.items = items;
    if (dueDate) updateData.dueDate = moment(dueDate).toDate();
    if (notes !== undefined) updateData.notes = notes;
    if (subtotal !== undefined) updateData.subtotal = subtotal;
    if (discount !== undefined) updateData.discount = discount;
    if (discountType !== undefined) updateData.discountType = discountType;
    if (tax !== undefined) updateData.tax = tax;
    if (totalAmount !== undefined) updateData.totalAmount = totalAmount;

    const invoice = await Invoice.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true }
    ).populate('studentId', 'studentName parentName email currency');

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    res.json({ message: 'Invoice updated successfully', invoice });
  } catch (error) {
    console.error('Update invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/invoices/:id/status', adminAuth, async (req, res) => {
  try {
    const { status, paidDate, enabled } = req.body;

    const updateData = { status };
    if (status === 'paid' && paidDate) {
      updateData.paidDate = moment(paidDate).toDate();
    }
    if (enabled !== undefined) {
      updateData.enabled = enabled;
    }

    const invoice = await Invoice.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true }
    ).populate('studentId', 'studentName parentName email');

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    res.json({ message: 'Invoice status updated successfully', invoice });
  } catch (error) {
    console.error('Update invoice status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/invoices/:id', adminAuth, async (req, res) => {
  try {
    const invoice = await Invoice.findById(req.params.id)
      .populate('studentId', 'studentName');

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    // Store deletion info before deleting
    const deletionInfo = {
      invoiceId: invoice._id,
      invoiceNumber: invoice.invoiceNumber,
      studentName: invoice.studentId?.studentName,
      totalAmount: invoice.totalAmount,
      currency: invoice.currency,
      deletedAt: new Date()
    };

    // Store in app locals for recent activity tracking
    if (!req.app.locals.deletedInvoices) {
      req.app.locals.deletedInvoices = [];
    }
    req.app.locals.deletedInvoices.unshift(deletionInfo);

    // Keep only last 20 deleted invoices in memory
    if (req.app.locals.deletedInvoices.length > 20) {
      req.app.locals.deletedInvoices = req.app.locals.deletedInvoices.slice(0, 20);
    }

    await Invoice.findByIdAndDelete(req.params.id);

    res.json({ message: 'Invoice deleted successfully' });
  } catch (error) {
    console.error('Delete invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/dashboard-stats', adminAuth, async (req, res) => {
  try {
    
    const currentMonth = moment().format('MM');
    const currentYear = moment().year();
    
    const startDate = moment().startOf('month').toDate();
    const endDate = moment().endOf('month').toDate();

    const totalStudents = await Student.countDocuments({ isActive: true, isDeleted: false });
    const totalTeachers = await Teacher.countDocuments({ isActive: true, isDeleted: false });
    
    const currentMonthClasses = await Class.countDocuments({
      scheduledDate: { $gte: startDate, $lte: endDate },
    });

    // Get paid invoices for this month (based on paidDate)
    const paidInvoicesThisMonth = await Invoice.find({
      status: 'paid',
      paidDate: { $gte: startDate, $lte: endDate }
    });

    // Currency conversion rates to USD
    const currencyToUSD = {
      'USD': 1,
      '$': 1,
      'GBP': 1.27,
      '£': 1.27,
      'EUR': 1.09,
      '€': 1.09,
      'AED': 0.27,
      'INR': 0.01127,
      '₹': 0.01127,
      'SAR': 0.27,
      'CAD': 0.74,
      'AUD': 0.65
    };

    // Calculate total revenue in USD
    const currentMonthRevenueUSD = paidInvoicesThisMonth.reduce((sum, invoice) => {
      const amount = parseFloat(invoice.totalAmount || 0);
      const currency = invoice.currency || 'USD';
      const conversionRate = currencyToUSD[currency] || 1;
      return sum + (amount * conversionRate);
    }, 0);

    const pendingInvoices = await Invoice.countDocuments({
      status: { $in: ['draft', 'sent'] }
    });

    const overdue = await Invoice.countDocuments({
      status: 'sent',
      dueDate: { $lt: new Date() }
    });

    // Get today's classes count
    const startOfToday = new Date();
    startOfToday.setHours(0, 0, 0, 0);
    const endOfToday = new Date();
    endOfToday.setHours(23, 59, 59, 999);

    const todayClasses = await Class.countDocuments({
      scheduledDate: {
        $gte: startOfToday,
        $lte: endOfToday
      }
    });

    // Get active trial requests count
    const activeTrials = await TrialRequest.countDocuments({
      status: { $in: ['pending', 'approved'] }
    });

    // Get students by country
    const studentsByCountry = await Student.aggregate([
      {
        $match: {
          isActive: true,
          isDeleted: false
        }
      },
      {
        $group: {
          _id: '$country',
          count: { $sum: 1 }
        }
      },
      {
        $sort: { count: -1 }
      },
      {
        $limit: 10
      }
    ]);

    // Map country data with colors and flags
    const countryColors = [
      '#1976d2', '#388e3c', '#f57c00', '#7b1fa2', '#d32f2f',
      '#0288d1', '#689f38', '#fbc02d', '#5d4037', '#455a64'
    ];

    const countryFlags = {
      'USA': '🇺🇸',
      'UK': '🇬🇧',
      'United Kingdom': '🇬🇧',
      'Canada': '🇨🇦',
      'Australia': '🇦🇺',
      'India': '🇮🇳',
      'UAE': '🇦🇪',
      'Saudi Arabia': '🇸🇦',
      'Pakistan': '🇵🇰',
      'Bangladesh': '🇧🇩',
      'Germany': '🇩🇪',
      'France': '🇫🇷',
      'Italy': '🇮🇹',
      'Spain': '🇪🇸',
      'Netherlands': '🇳🇱',
      'Singapore': '🇸🇬',
      'Malaysia': '🇲🇾',
      'Ireland': '🇮🇪',
      'New Zealand': '🇳🇿',
      'South Africa': '🇿🇦'
    };

    const formattedStudentsByCountry = studentsByCountry.map((item, index) => ({
      country: item._id || 'Unknown',
      count: item.count,
      flag: countryFlags[item._id] || '🌍',
      color: countryColors[index % countryColors.length]
    }));

    // Get recent activities from the database (last 7 days)
    const activityTimeframe = moment().subtract(7, 'days').toDate();

    // Completed classes
    const recentClasses = await Class.find({
      status: 'completed',
      updatedAt: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .populate('teacherId', 'name')
    .sort({ updatedAt: -1 })
    .limit(5);

    // Created classes
    const recentClassCreations = await Class.find({
      createdAt: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .populate('teacherId', 'name')
    .sort({ createdAt: -1 })
    .limit(5);

    // Deleted classes (soft delete - check for isDeleted flag)
    const recentClassDeletions = await Class.find({
      isDeleted: true,
      updatedAt: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .populate('teacherId', 'name')
    .sort({ updatedAt: -1 })
    .limit(3);

    // Student registrations
    const recentStudents = await Student.find({
      createdAt: { $gte: activityTimeframe }
    })
    .sort({ createdAt: -1 })
    .limit(5);

    // Teacher registrations
    const recentTeachers = await Teacher.find({
      createdAt: { $gte: activityTimeframe }
    })
    .sort({ createdAt: -1 })
    .limit(5);

    // Invoice creations
    const recentInvoiceCreations = await Invoice.find({
      createdAt: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .sort({ createdAt: -1 })
    .limit(5);

    // Invoice payments
    const recentPayments = await Invoice.find({
      status: 'paid',
      paidDate: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .sort({ paidDate: -1 })
    .limit(4);

    // Invoice updates (status changes)
    const recentInvoiceUpdates = await Invoice.find({
      updatedAt: { $gte: activityTimeframe },
      createdAt: { $lt: moment().subtract(1, 'minute').toDate() } // Exclude just created invoices
    })
    .populate('studentId', 'studentName')
    .sort({ updatedAt: -1 })
    .limit(3);

    // Trial request creations
    const recentTrialRequests = await TrialRequest.find({
      createdAt: { $gte: activityTimeframe }
    })
    .sort({ createdAt: -1 })
    .limit(4);

    // Trial request updates (status changes)
    const recentTrialUpdates = await TrialRequest.find({
      updatedAt: { $gte: activityTimeframe },
      createdAt: { $lt: moment().subtract(1, 'minute').toDate() } // Exclude just created requests
    })
    .sort({ updatedAt: -1 })
    .limit(3);

    // Reschedule request creations
    const recentRescheduleRequests = await RescheduleRequest.find({
      createdAt: { $gte: activityTimeframe }
    })
    .populate('studentId', 'studentName')
    .sort({ createdAt: -1 })
    .limit(4);

    // Reschedule request updates (status changes)
    const recentRescheduleUpdates = await RescheduleRequest.find({
      updatedAt: { $gte: activityTimeframe },
      status: { $in: ['approved', 'rejected'] }
    })
    .populate('studentId', 'studentName')
    .sort({ updatedAt: -1 })
    .limit(3);

    const recentActivity = [];

    // Add recent class completions
    recentClasses.forEach(cls => {
      recentActivity.push({
        id: `class-complete-${cls._id}`,
        type: 'class_complete',
        title: 'Class Completed',
        description: `${cls.subject} class with ${cls.studentId?.studentName || 'Unknown Student'}`,
        time: moment(cls.updatedAt).fromNow(),
        timestamp: cls.updatedAt,
        icon: 'EventNote',
        color: 'success'
      });
    });

    // Add recent class creations
    recentClassCreations.forEach(cls => {
      recentActivity.push({
        id: `class-create-${cls._id}`,
        type: 'class_create',
        title: 'Class Scheduled',
        description: `${cls.subject} class scheduled for ${cls.studentId?.studentName || 'Unknown Student'}`,
        time: moment(cls.createdAt).fromNow(),
        timestamp: cls.createdAt,
        icon: 'Schedule',
        color: 'primary'
      });
    });

    // Add recent class deletions
    recentClassDeletions.forEach(cls => {
      recentActivity.push({
        id: `class-delete-${cls._id}`,
        type: 'class_delete',
        title: 'Class Cancelled',
        description: `${cls.subject} class cancelled for ${cls.studentId?.studentName || 'Unknown Student'}`,
        time: moment(cls.updatedAt).fromNow(),
        timestamp: cls.updatedAt,
        icon: 'EventBusy',
        color: 'error'
      });
    });

    // Add recent student registrations
    recentStudents.forEach(student => {
      recentActivity.push({
        id: `student-${student._id}`,
        type: 'student_register',
        title: 'New Student',
        description: `${student.studentName} registered`,
        time: moment(student.createdAt).fromNow(),
        timestamp: student.createdAt,
        icon: 'PersonAdd',
        color: 'info'
      });
    });

    // Add recent teacher registrations
    recentTeachers.forEach(teacher => {
      recentActivity.push({
        id: `teacher-${teacher._id}`,
        type: 'teacher_register',
        title: 'New Teacher',
        description: `${teacher.name} joined the team`,
        time: moment(teacher.createdAt).fromNow(),
        timestamp: teacher.createdAt,
        icon: 'GroupAdd',
        color: 'secondary'
      });
    });

    // Add recent invoice creations
    recentInvoiceCreations.forEach(invoice => {
      recentActivity.push({
        id: `invoice-create-${invoice._id}`,
        type: 'invoice_create',
        title: 'Invoice Created',
        description: `Invoice ${invoice.invoiceNumber} generated for ${invoice.studentId?.studentName || 'Student'}`,
        time: moment(invoice.createdAt).fromNow(),
        timestamp: invoice.createdAt,
        icon: 'Receipt',
        color: 'info'
      });
    });

    // Add recent payments
    recentPayments.forEach(invoice => {
      recentActivity.push({
        id: `payment-${invoice._id}`,
        type: 'payment',
        title: 'Payment Received',
        description: `${invoice.currency || '£'}${invoice.totalAmount} from ${invoice.studentId?.studentName || 'Student'}`,
        time: moment(invoice.paidDate).fromNow(),
        timestamp: invoice.paidDate,
        icon: 'MonetizationOn',
        color: 'success'
      });
    });

    // Add recent invoice updates
    recentInvoiceUpdates.forEach(invoice => {
      recentActivity.push({
        id: `invoice-update-${invoice._id}`,
        type: 'invoice_update',
        title: 'Invoice Updated',
        description: `Invoice ${invoice.invoiceNumber} status changed to ${invoice.status}`,
        time: moment(invoice.updatedAt).fromNow(),
        timestamp: invoice.updatedAt,
        icon: 'Receipt',
        color: 'warning'
      });
    });

    // Add recent invoice deletions (from app locals)
    const deletedInvoices = req.app.locals.deletedInvoices || [];
    deletedInvoices
      .filter(invoice => moment(invoice.deletedAt).isAfter(activityTimeframe))
      .slice(0, 3)
      .forEach(invoice => {
        recentActivity.push({
          id: `invoice-delete-${invoice.invoiceId}`,
          type: 'invoice_delete',
          title: 'Invoice Deleted',
          description: `Invoice ${invoice.invoiceNumber} deleted for ${invoice.studentName || 'Student'}`,
          time: moment(invoice.deletedAt).fromNow(),
          timestamp: invoice.deletedAt,
          icon: 'Delete',
          color: 'error'
        });
      });

    // Add recent trial requests
    recentTrialRequests.forEach(trial => {
      recentActivity.push({
        id: `trial-${trial._id}`,
        type: 'trial_request',
        title: 'Trial Request',
        description: `New trial request from ${trial.studentName}`,
        time: moment(trial.createdAt).fromNow(),
        timestamp: trial.createdAt,
        icon: 'Assignment',
        color: 'info'
      });
    });

    // Add recent trial updates
    recentTrialUpdates.forEach(trial => {
      recentActivity.push({
        id: `trial-update-${trial._id}`,
        type: 'trial_update',
        title: 'Trial Request Updated',
        description: `Trial request for ${trial.studentName} - ${trial.status}`,
        time: moment(trial.updatedAt).fromNow(),
        timestamp: trial.updatedAt,
        icon: 'AssignmentTurnedIn',
        color: 'primary'
      });
    });

    // Add recent reschedule requests
    recentRescheduleRequests.forEach(request => {
      recentActivity.push({
        id: `reschedule-${request._id}`,
        type: 'reschedule_request',
        title: 'Reschedule Request',
        description: `${request.studentId?.studentName || 'Student'} requested to reschedule ${request.classDetails?.subject || 'class'}`,
        time: moment(request.createdAt).fromNow(),
        timestamp: request.createdAt,
        icon: 'AccessTime',
        color: 'warning'
      });
    });

    // Add recent reschedule updates
    recentRescheduleUpdates.forEach(request => {
      recentActivity.push({
        id: `reschedule-update-${request._id}`,
        type: 'reschedule_update',
        title: 'Reschedule Request Updated',
        description: `Reschedule request ${request.status} for ${request.studentId?.studentName || 'Student'}`,
        time: moment(request.updatedAt).fromNow(),
        timestamp: request.updatedAt,
        icon: 'Update',
        color: request.status === 'approved' ? 'success' : 'error'
      });
    });

    // Sort all activities by timestamp (most recent first) and take top 15
    recentActivity.sort((a, b) => {
      return new Date(b.timestamp) - new Date(a.timestamp);
    });

    // Get monthly stats for the last 6 months
    const monthlyClassStats = [];
    const monthlyRevenueStats = [];

    for (let i = 5; i >= 0; i--) {
      const monthStart = moment().subtract(i, 'months').startOf('month').toDate();
      const monthEnd = moment().subtract(i, 'months').endOf('month').toDate();

      const monthClasses = await Class.countDocuments({
        scheduledDate: { $gte: monthStart, $lte: monthEnd },
        status: 'completed'
      });

      const monthRevenue = await Class.aggregate([
        {
          $match: {
            scheduledDate: { $gte: monthStart, $lte: monthEnd },
            status: 'completed',
            'attendanceStatus.studentAttended': true
          }
        },
        {
          $lookup: {
            from: 'students',
            localField: 'studentId',
            foreignField: '_id',
            as: 'student'
          }
        },
        {
          $unwind: '$student'
        },
        {
          $group: {
            _id: null,
            totalRevenue: {
              $sum: {
                $multiply: [
                  { $divide: ['$duration', 60] },
                  '$student.hourlyRate'
                ]
              }
            }
          }
        }
      ]);

      monthlyClassStats.push(monthClasses);
      monthlyRevenueStats.push(monthRevenue[0]?.totalRevenue || 0);
    }

    res.json({
      totalStudents,
      totalTeachers,
      totalClasses: currentMonthClasses,
      totalRevenue: parseFloat(currentMonthRevenueUSD.toFixed(2)),
      todayClasses,
      activeTrials,
      pendingInvoices,
      overdueInvoices: overdue,
      studentsByCountry: formattedStudentsByCountry,
      recentActivity: recentActivity.slice(0, 15),
      monthlyStats: {
        classes: monthlyClassStats,
        revenue: monthlyRevenueStats
      }
    });
  } catch (error) {
    console.error('Get dashboard stats error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

export default router;