import mongoose from 'mongoose';

const answerSchema = new mongoose.Schema({
  questionId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Question',
    required: true
  },
  selectedOptionId: {
    type: String,
    required: true
  },
  isCorrect: {
    type: Boolean,
    default: false
  },
  marksObtained: {
    type: Number,
    default: 0
  }
});

const submissionSchema = new mongoose.Schema({
  examId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Exam',
    required: true
  },
  studentId: {
    type: String,
    required: true
  },
  answers: [answerSchema],
  totalScore: {
    type: Number,
    default: 0
  },
  percentage: {
    type: Number,
    default: 0
  },
  passed: {
    type: Boolean,
    default: false
  },
  status: {
    type: String,
    enum: ['in-progress', 'submitted', 'evaluated'],
    default: 'in-progress'
  },
  startedAt: {
    type: Date,
    default: Date.now
  },
  submittedAt: {
    type: Date
  },
  timeTaken: {
    type: Number,
    default: 0
  }
});

submissionSchema.index({ examId: 1, studentId: 1 }, { unique: true });

submissionSchema.methods.calculateScore = async function() {
  const Question = mongoose.model('Question');
  const Exam = mongoose.model('Exam');
  
  const exam = await Exam.findById(this.examId);
  const questions = await Question.find({ examId: this.examId });
  
  let totalScore = 0;
  
  for (const answer of this.answers) {
    const question = questions.find(q => q._id.toString() === answer.questionId.toString());
    if (question) {
      const correctOption = question.options.find(opt => opt.isCorrect);
      if (correctOption && correctOption.optionId === answer.selectedOptionId) {
        answer.isCorrect = true;
        answer.marksObtained = question.marks;
        totalScore += question.marks;
      }
    }
  }
  
  this.totalScore = totalScore;
  this.percentage = (totalScore / exam.totalMarks) * 100;
  this.passed = this.totalScore >= exam.passingMarks;
  this.status = 'evaluated';
  
  if (this.submittedAt && this.startedAt) {
    this.timeTaken = Math.floor((this.submittedAt - this.startedAt) / 1000);
  }
  
  return this.save();
};

export default mongoose.model('Submission', submissionSchema);