const { body, param, query, validationResult } = require('express-validator');
const xss = require('xss');
const mongoSanitize = require('express-mongo-sanitize');
const hpp = require('hpp');
const auditLogger = require('../utils/auditLogger');

class EnterpriseValidation {
  constructor() {
    this.xssOptions = {
      whiteList: {
        // Only allow very specific HTML tags if needed
        'b': [],
        'i': [],
        'em': [],
        'strong': [],
        'p': ['class'],
        'br': []
      },
      stripIgnoreTag: true,
      stripIgnoreTagBody: ['script']
    };
  }

  // Input sanitization middleware
  sanitizeInput() {
    return (req, res, next) => {
      // Sanitize against NoSQL injection
      mongoSanitize.sanitize(req.body);
      mongoSanitize.sanitize(req.query);
      mongoSanitize.sanitize(req.params);

      // Protect against HTTP Parameter Pollution
      hpp()(req, res, () => {
        // XSS protection for all string inputs
        this.sanitizeObjectRecursive(req.body);
        this.sanitizeObjectRecursive(req.query);
        this.sanitizeObjectRecursive(req.params);

        next();
      });
    };
  }

  // Recursive XSS sanitization
  sanitizeObjectRecursive(obj) {
    if (typeof obj === 'string') {
      return xss(obj, this.xssOptions);
    }

    if (Array.isArray(obj)) {
      return obj.map(item => this.sanitizeObjectRecursive(item));
    }

    if (obj && typeof obj === 'object') {
      for (const key in obj) {
        if (obj.hasOwnProperty(key)) {
          obj[key] = this.sanitizeObjectRecursive(obj[key]);
        }
      }
    }

    return obj;
  }

  // Validation error handler
  handleValidationErrors() {
    return (req, res, next) => {
      const errors = validationResult(req);
      
      if (!errors.isEmpty()) {
        const errorDetails = errors.array().map(error => ({
          field: error.path || error.param,
          message: error.msg,
          value: error.value
        }));

        auditLogger.warn('Validation failed', {
          ip: req.ip,
          endpoint: req.path,
          method: req.method,
          errors: errorDetails,
          userId: req.user?.id
        });

        return res.status(400).json({
          error: 'Validation failed',
          details: errorDetails
        });
      }
      
      next();
    };
  }

  // User authentication validation
  validateAuth() {
    return [
      body('username')
        .isEmail()
        .normalizeEmail()
        .isLength({ min: 5, max: 100 })
        .withMessage('Valid email address required'),
      
      body('password')
        .isLength({ min: 8, max: 128 })
        .withMessage('Password must be 8-128 characters')
        .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/)
        .withMessage('Password must contain uppercase, lowercase, number and special character'),
      
      this.handleValidationErrors()
    ];
  }

  // User registration validation
  validateUserRegistration() {
    return [
      body('username')
        .isEmail()
        .normalizeEmail()
        .isLength({ min: 5, max: 100 })
        .withMessage('Valid email address required'),
      
      body('password')
        .isLength({ min: 12, max: 128 })
        .withMessage('Password must be 12-128 characters')
        .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/)
        .withMessage('Password must contain uppercase, lowercase, number and special character'),
      
      body('confirmPassword')
        .custom((value, { req }) => {
          if (value !== req.body.password) {
            throw new Error('Passwords do not match');
          }
          return true;
        }),
      
      body('userType')
        .isIn(['admin', 'teacher', 'student'])
        .withMessage('Invalid user type'),
      
      body('firstName')
        .isLength({ min: 1, max: 50 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('First name must contain only letters'),
      
      body('lastName')
        .isLength({ min: 1, max: 50 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Last name must contain only letters'),
      
      body('phone')
        .optional()
        .isMobilePhone()
        .withMessage('Invalid phone number'),
      
      this.handleValidationErrors()
    ];
  }

  // Student creation validation
  validateStudentCreation() {
    return [
      body('studentName')
        .isLength({ min: 2, max: 100 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Student name must contain only letters and spaces'),
      
      body('email')
        .isEmail()
        .normalizeEmail()
        .withMessage('Valid email required'),
      
      body('grade')
        .isLength({ min: 1, max: 20 })
        .withMessage('Grade is required'),
      
      body('subjects')
        .isArray({ min: 1, max: 10 })
        .withMessage('At least one subject required'),
      
      body('subjects.*')
        .isLength({ min: 2, max: 50 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Subject names must contain only letters'),
      
      body('guardianName')
        .isLength({ min: 2, max: 100 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Guardian name must contain only letters'),
      
      body('guardianPhone')
        .isMobilePhone()
        .withMessage('Valid guardian phone number required'),
      
      body('guardianEmail')
        .optional()
        .isEmail()
        .normalizeEmail()
        .withMessage('Valid guardian email required'),
      
      body('hourlyRate')
        .isFloat({ min: 0, max: 1000 })
        .withMessage('Hourly rate must be between 0 and 1000'),
      
      body('currency')
        .isIn(['GBP', 'USD', 'EUR'])
        .withMessage('Invalid currency'),
      
      this.handleValidationErrors()
    ];
  }

  // Teacher creation validation
  validateTeacherCreation() {
    return [
      body('name')
        .isLength({ min: 2, max: 100 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Teacher name must contain only letters and spaces'),
      
      body('email')
        .isEmail()
        .normalizeEmail()
        .withMessage('Valid email required'),
      
      body('subjects')
        .isArray({ min: 1, max: 10 })
        .withMessage('At least one subject required'),
      
      body('subjects.*')
        .isLength({ min: 2, max: 50 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Subject names must contain only letters'),
      
      body('qualifications')
        .isArray({ min: 1, max: 10 })
        .withMessage('At least one qualification required'),
      
      body('qualifications.*')
        .isLength({ min: 2, max: 100 })
        .withMessage('Qualification must be 2-100 characters'),
      
      body('experience')
        .isLength({ min: 1, max: 50 })
        .withMessage('Experience is required'),
      
      body('hourlyRate')
        .optional()
        .isFloat({ min: 0, max: 1000 })
        .withMessage('Hourly rate must be between 0 and 1000'),
      
      body('phone')
        .optional()
        .isMobilePhone()
        .withMessage('Invalid phone number'),
      
      this.handleValidationErrors()
    ];
  }

  // Class scheduling validation
  validateClassScheduling() {
    return [
      body('studentId')
        .isMongoId()
        .withMessage('Valid student ID required'),
      
      body('teacherId')
        .isMongoId()
        .withMessage('Valid teacher ID required'),
      
      body('subject')
        .isLength({ min: 2, max: 50 })
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Subject must contain only letters'),
      
      body('scheduledDate')
        .isISO8601()
        .withMessage('Valid date required')
        .custom((value) => {
          const date = new Date(value);
          const now = new Date();
          if (date <= now) {
            throw new Error('Scheduled date must be in the future');
          }
          return true;
        }),
      
      body('duration')
        .isInt({ min: 15, max: 480 })
        .withMessage('Duration must be between 15 and 480 minutes'),
      
      body('notes')
        .optional()
        .isLength({ max: 500 })
        .withMessage('Notes must not exceed 500 characters'),
      
      this.handleValidationErrors()
    ];
  }

  // File upload validation
  validateFileUpload() {
    return [
      body('description')
        .optional()
        .isLength({ max: 200 })
        .withMessage('Description must not exceed 200 characters'),
      
      this.handleValidationErrors()
    ];
  }

  // ID parameter validation
  validateMongoId() {
    return [
      param('id')
        .isMongoId()
        .withMessage('Valid ID required'),
      
      this.handleValidationErrors()
    ];
  }

  // Pagination validation
  validatePagination() {
    return [
      query('page')
        .optional()
        .isInt({ min: 1, max: 1000 })
        .withMessage('Page must be between 1 and 1000'),
      
      query('limit')
        .optional()
        .isInt({ min: 1, max: 100 })
        .withMessage('Limit must be between 1 and 100'),
      
      query('sortBy')
        .optional()
        .isLength({ min: 1, max: 50 })
        .matches(/^[a-zA-Z_]+$/)
        .withMessage('Invalid sort field'),
      
      query('sortOrder')
        .optional()
        .isIn(['asc', 'desc'])
        .withMessage('Sort order must be asc or desc'),
      
      this.handleValidationErrors()
    ];
  }

  // Search validation
  validateSearch() {
    return [
      query('q')
        .optional()
        .isLength({ min: 1, max: 100 })
        .matches(/^[a-zA-Z0-9\s@._-]+$/)
        .withMessage('Search query contains invalid characters'),
      
      query('filter')
        .optional()
        .isLength({ min: 1, max: 50 })
        .matches(/^[a-zA-Z_]+$/)
        .withMessage('Invalid filter field'),
      
      this.handleValidationErrors()
    ];
  }

  // Date range validation
  validateDateRange() {
    return [
      query('startDate')
        .optional()
        .isISO8601()
        .withMessage('Valid start date required'),
      
      query('endDate')
        .optional()
        .isISO8601()
        .withMessage('Valid end date required')
        .custom((value, { req }) => {
          if (req.query.startDate && value) {
            const start = new Date(req.query.startDate);
            const end = new Date(value);
            if (end <= start) {
              throw new Error('End date must be after start date');
            }
          }
          return true;
        }),
      
      this.handleValidationErrors()
    ];
  }

  // Password change validation
  validatePasswordChange() {
    return [
      body('currentPassword')
        .isLength({ min: 1 })
        .withMessage('Current password required'),
      
      body('newPassword')
        .isLength({ min: 12, max: 128 })
        .withMessage('New password must be 12-128 characters')
        .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/)
        .withMessage('New password must contain uppercase, lowercase, number and special character')
        .custom((value, { req }) => {
          if (value === req.body.currentPassword) {
            throw new Error('New password must be different from current password');
          }
          return true;
        }),
      
      body('confirmPassword')
        .custom((value, { req }) => {
          if (value !== req.body.newPassword) {
            throw new Error('Password confirmation does not match');
          }
          return true;
        }),
      
      this.handleValidationErrors()
    ];
  }

  // Custom validation for business rules
  validateBusinessRules() {
    return (req, res, next) => {
      // Add custom business logic validation here
      // Example: Check if user has permission to perform action
      
      if (req.user && req.body) {
        // Prevent users from modifying their own userType
        if (req.body.userType && req.user.userType !== 'admin') {
          return res.status(403).json({
            error: 'Insufficient permissions to modify user type'
          });
        }
      }
      
      next();
    };
  }
}

module.exports = new EnterpriseValidation();